import yt_dlp
import os
import asyncio
import aiohttp
from config import DOWNLOAD_PATH, YOUTUBE_API_KEY

os.makedirs(DOWNLOAD_PATH, exist_ok=True)

COOKIES_FILE = "/home/downloadultraalm/public_html/cookies.txt"
FFMPEG_PATH = os.path.expanduser("~/bin/ffmpeg")

BASE_OPTS = {
    "ffmpeg_location": FFMPEG_PATH,
    "quiet": True,
    "no_warnings": True,
    "http_headers": {
        "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36",
    },
}

def is_youtube(url: str) -> bool:
    return "youtube.com" in url or "youtu.be" in url

def get_video_id(url: str) -> str:
    import re
    patterns = [r"(?:v=|youtu\.be/|shorts/)([a-zA-Z0-9_-]{11})"]
    for p in patterns:
        m = re.search(p, url)
        if m:
            return m.group(1)
    return ""

def get_platform(url: str) -> str:
    platforms = {
        "youtube.com": "يوتيوب", "youtu.be": "يوتيوب",
        "tiktok.com": "تيك توك", "instagram.com": "انستجرام",
        "facebook.com": "فيسبوك", "twitter.com": "تويتر",
        "x.com": "تويتر", "reddit.com": "ريديت",
        "vimeo.com": "Vimeo", "twitch.tv": "Twitch",
        "dailymotion.com": "Dailymotion", "pinterest.com": "Pinterest"
    }
    for domain, name in platforms.items():
        if domain in url:
            return name
    return "موقع آخر"

async def get_youtube_info(video_id: str) -> dict:
    url = (
        f"https://www.googleapis.com/youtube/v3/videos"
        f"?id={video_id}&key={YOUTUBE_API_KEY}"
        f"&part=snippet,contentDetails"
    )
    async with aiohttp.ClientSession() as session:
        async with session.get(url) as resp:
            data = await resp.json()
    items = data.get("items", [])
    if not items:
        raise Exception("الفيديو غير موجود أو محذوف")
    item = items[0]
    snippet = item["snippet"]
    import re
    duration_str = item["contentDetails"]["duration"]
    m = re.match(r"PT(?:(\d+)H)?(?:(\d+)M)?(?:(\d+)S)?", duration_str)
    hours = int(m.group(1) or 0)
    mins  = int(m.group(2) or 0)
    secs  = int(m.group(3) or 0)
    total_seconds = hours * 3600 + mins * 60 + secs
    return {
        "title": snippet["title"],
        "uploader": snippet["channelTitle"],
        "thumbnail": snippet["thumbnails"]["high"]["url"],
        "duration": total_seconds,
        "video_id": video_id,
    }

async def download_youtube(video_id: str, quality: str, audio_only: bool) -> dict:
    url = f"https://www.youtube.com/watch?v={video_id}"
    output_template = os.path.join(DOWNLOAD_PATH, "%(id)s.%(ext)s")
    quality_map = {
        "2160": "bestvideo[height<=2160]+bestaudio/best",
        "1080": "bestvideo[height<=1080]+bestaudio/best",
        "720":  "bestvideo[height<=720]+bestaudio/best",
        "480":  "bestvideo[height<=480]+bestaudio/best",
        "360":  "bestvideo[height<=360]+bestaudio/best",
        "best": "bestvideo+bestaudio/best",
    }
    if audio_only:
        ydl_opts = {
            **BASE_OPTS,
            "cookiefile": COOKIES_FILE,
            "format": "bestaudio/best",
            "outtmpl": output_template,
            "postprocessors": [{
                "key": "FFmpegExtractAudio",
                "preferredcodec": "mp3",
                "preferredquality": "320",
            }],
        }
    else:
        ydl_opts = {
            **BASE_OPTS,
            "cookiefile": COOKIES_FILE,
            "format": quality_map.get(quality, "bestvideo+bestaudio/best"),
            "outtmpl": output_template,
            "merge_output_format": "mp4",
            "extractor_args": {
                "youtube": {
                    "player_client": ["default", "-android_sdkless"],
                    "formats": ["missing_pot"],
                }
            },
            "postprocessors": [{
                "key": "FFmpegVideoConvertor",
                "preferedformat": "mp4",
            }],
        }
    loop = asyncio.get_event_loop()
    def _download():
        with yt_dlp.YoutubeDL(ydl_opts) as ydl:
            info = ydl.extract_info(url, download=True)
            filename = ydl.prepare_filename(info)
            if audio_only:
                filename = filename.rsplit(".", 1)[0] + ".mp3"
            return {
                "title": info.get("title", "غير معروف"),
                "filename": filename,
                "filesize": os.path.getsize(filename) / (1024 * 1024)
            }
    return await loop.run_in_executor(None, _download)

async def get_video_info(url: str) -> dict:
    if is_youtube(url):
        video_id = get_video_id(url)
        if not video_id:
            raise Exception("رابط يوتيوب غير صحيح")
        return await get_youtube_info(video_id)
    ydl_opts = {**BASE_OPTS}
    loop = asyncio.get_event_loop()
    def _extract():
        with yt_dlp.YoutubeDL(ydl_opts) as ydl:
            return ydl.extract_info(url, download=False)
    return await loop.run_in_executor(None, _extract)

async def download_video(url: str, quality: str = "best", audio_only: bool = False) -> dict:
    if is_youtube(url):
        video_id = get_video_id(url)
        return await download_youtube(video_id, quality, audio_only)
    output_template = os.path.join(DOWNLOAD_PATH, "%(id)s.%(ext)s")
    quality_map = {
        "2160": "bestvideo[height<=2160]+bestaudio/best",
        "1080": "bestvideo[height<=1080]+bestaudio/best",
        "720":  "bestvideo[height<=720]+bestaudio/best",
        "480":  "bestvideo[height<=480]+bestaudio/best",
        "360":  "bestvideo[height<=360]+bestaudio/best",
        "best": "bestvideo+bestaudio/best"
    }
    if audio_only:
        ydl_opts = {
            **BASE_OPTS,
            "format": "bestaudio/best",
            "outtmpl": output_template,
            "postprocessors": [{
                "key": "FFmpegExtractAudio",
                "preferredcodec": "mp3",
                "preferredquality": "320",
            }],
        }
    else:
        ydl_opts = {
            **BASE_OPTS,
            "format": quality_map.get(quality, "bestvideo+bestaudio/best"),
            "outtmpl": output_template,
            "merge_output_format": "mp4",
            "postprocessors": [{
                "key": "FFmpegVideoConvertor",
                "preferedformat": "mp4",
            }],
        }
    loop = asyncio.get_event_loop()
    def _download():
        with yt_dlp.YoutubeDL(ydl_opts) as ydl:
            info = ydl.extract_info(url, download=True)
            filename = ydl.prepare_filename(info)
            if audio_only:
                filename = filename.rsplit(".", 1)[0] + ".mp3"
            return {
                "title": info.get("title", "غير معروف"),
                "duration": info.get("duration", 0),
                "filename": filename,
                "filesize": os.path.getsize(filename) / (1024 * 1024)
            }
    return await loop.run_in_executor(None, _download)
